# ======================================================================
# PURPOSE
# =========
# The purpose of this script is to replicate all Figures in the
# Olson et al. Journal of Statistical Software manuscript.

# STATUS
# ========
# Working

# INPUT
# =======
# Relevant datasets in the stilt package.

# NOTE
# ========
# The results are reproducible on R > 3.2.4 under Windows, Mac OS, and Linux, with extremely
# minor numerical errors. 

# USASE
# ========
# source("replication.R")

# OUTPUT
# ========
# All figures for the Olson et al. JSS manuscript.

# REQUIRES
# =========
# R (3.2.4) or higher, stilt.

# HISTORY
# ========
# Jul 11 2017 Written by Roman Olson, Yonsei University.
# ======================================================================

# SETTING UP THE ENVIRONMENT (CAN BE COMMENTED OUT!!!)
# ===========================
#install.packages("stilt")
#library(stilt)

## FIGURE 1
## ==========
cat("===========\n")
cat("FIGURE 1\n")
cat("===========\n")
pdf("Figure1.pdf", width=6, height=6, onefile=FALSE)
data("Data.1D.par")
data("Data.1D.model")
plot(NA, xlim=c(0, 11), ylim=c(-150, 150), xlab="Time", ylab="Model output")
for (c in 1:21) {lines(Data.1D.model$t, Data.1D.model$out[,c])}
dev.off()

## FITTING THE EMULATOR TO TOY DATASET AND FIGURE 2
## ================================================
cat("===========\n")
cat("TOY DATASET EMULATOR AND FIGURE 2\n")
cat("===========\n")
emul1D = emulator(Data.1D.par, Data.1D.model, par.reg=FALSE, time.reg=TRUE,
kappa0=100, zeta0=100)
pdf("Figure2.pdf", width=8, height=6, onefile=FALSE)
test.all(emul1D, 9)
dev.off()

##  FIGURE 3
## ============================================================
cat("===========\n")
cat("FIGURE 3\n")
cat("===========\n")
data("Data.AR1Korea.model")
data("Data.AR1Korea.par")
mycolors = rainbow(29)
pdf("Figure3.pdf", width=6, height=6, onefile=FALSE)
plot.default(NA, xlim=c(2081, 2100), ylim=c(0,10), xlab="Year",
             ylab="JJA Mean Max Temp Anomaly wrt. 1973-2005 [K]")
for (c in 1:29) {lines(Data.AR1Korea.model$t, Data.AR1Korea.model$out[,c],
                       col=mycolors[c])}
dev.off()

## FITTING THE EMULATOR TO THE KOREAN TEMPERATURES AND FIGURE 4
## =============================================================
cat("===========\n")
cat("KOREAN TEMP EMULATOR AND FIGURE 4\n")
cat("===========\n")
emul = emulator(Data.AR1Korea.par, Data.AR1Korea.model, par.reg=c(TRUE,
FALSE), time.reg=TRUE, kappa0=1, zeta0=1, myrel.tol=1E-9, fix.betas=FALSE)

pred = emul.predict(emul, c(1, 0))
pdf("Figure4.pdf", width=6, height=6, onefile=FALSE)
plot.default(NA, xlim=c(2081, 2100), ylim=c(3.5,7.5), xlab="Year",
             ylab="JJA Mean Max Temp Anomaly wrt. 1973-2005 [K]")
lines(emul$t.vec, pred$mean)
lines(emul$t.vec, pred$mean + sqrt(diag(pred$covariance)), lty=2)
lines(emul$t.vec, pred$mean - sqrt(diag(pred$covariance)), lty=2)
dev.off()

## CROSS-VALIDATING SICOPOLIS EMULATOR: FIGURE 5
cat("===========\n")
cat("FIGURE 5\n")
cat("===========\n")
data("emul.Sicopolis")
pdf("Figure5.pdf", width=6, height=6, onefile=FALSE)
test.csv(emul.Sicopolis, num.test=3, plot.std=TRUE, theseed=13241240)
dev.off()

cat("===========\n")
cat("SICOPOLIS EMULATOR COVERAGE\n")
cat("===========\n")
mytest = test.csv(emul.Sicopolis, num.test=10, plot.std=FALSE, theseed=13241240,
                  make.plot=FALSE)
cat("95% CI coverage:", mytest$coverage, "\n")


## RESPONSE SURFACE PLOT: FIGURE 6
cat("===========\n")
cat("FIGURE 6\n")
cat("===========\n")
pdf("Figure6.pdf", width=6, height=6, onefile=FALSE)
rsurface.plot(emul.Sicopolis, parind=c(4,5), parvals=c(3, 10, 45, NA, NA),
              tind=600, n1=10, n2=10)
dev.off()
